	TITLE   "Cardreader with PIC16C84"
	LIST    P=PIC16C84, R=HEX
	INCLUDE "PICREG.EQU"

; PIC16C84 Pin assignment

MR_CLK          equ     0       ; i   RB0 = Magnetic reader Clock
MR_DATA         equ     7       ; i   RB7 = Magnetic reader Data


; PIC16C84 RAM Register Assignments
MR_PORT         equ     PORTB

; Flags in Information Byte
BFR_FULL        equ     02h     ; Data is in new in block
FAIL            equ     03h     ; Indicates error in just read block
ES_FOUND	equ	04h	; Have found End Sentinel. Should stop reading.

; RAM memory allocations
	cblock	0x0C
		CINFO		; Information Byte with Flags
		TEMP_W		; Temporary W Save Address
		TEMP_S		; Temporary STATUS Save Address
		TIME		; Down counting of microseconds
		LOOPCNT		; General loopcounter

		CARD_STATE	; State for card reading algorithm
		READ		; Current read data nibble/byte
		TIME_OUT	; Time out counter
		CHECKSUM	; For calculating the checksum in run
				; Upper nibble is checksum, lower parity
		BFR_PTR		; Pointer to were we are in buffer

		BFR_START	; Start of buffer
		BFR2		; MPASM seems broken so to allocate buffer
		BFR3		; I'll do like this
		BFR4
		BFR5
		BFR6
		BFR7
		BFR_END		; Buffer (and memory) ends here
	endc


	org     PIC84           ; Reset vector
	goto    INIT            ; Jump to initialization routine

	org     INTVEC          ; Interrupt vector
RESET_GIE:
	bcf	INTCON,GIE	; This is what the 16C84 manual tells me
	btfsc	INTCON,GIE	; to do
	goto	RESET_GIE

	push                    ; Save registers
	call    INTMAIN         ; Call main interrupt routine
	pop                     ; Restore registers
	retfie                  ; Return from interrupt and clear flag

	include "display.inc"	

	cblock			; Check so we don't run out of RAM
		LASTPOS
	endc
	if (LASTPOS > 0x2F) 
		ERROR "RAM allocation oversized"
	endif

BOOTMSG1:
	DEFSTR	"CardRead"	; Boot message
BOOTMSG2:
	DEFSTR	"er V 0.4"
ERRORMSG1:
	DEFSTR	"Read Err"	; Failure message
ERRORMSG2:
	DEFSTR	"or !!   "
 
INIT:
	clrf    RTCC            ; Clear RTCC
	movlw   b'10110000'     ; Set interrupt = ON, RT = ON, INT = ON
	movwf   INTCON          ; Store in INTCON register
	clrf    PORTA           ; Clear port A register
	clrf    PORTB           ; Clear port B register
	bsf     STATUS,RP0      ; Must be in page 1 to access OPTION register
	movlw   b'11010001'     ; Set RTCC to osc/4 & 1:4 presc, rising edge
	movwf   OPTREG          ; Store in OPTION register
	movlw	b'11100000'	; Set PORTA Tristate Latches (1=Input)
	movwf	TRISA		; Store in PORTA tristate register
	movlw   b'11110001'     ; Set PORTB Tristate Latches
	movwf   TRISB           ; Store in PORTB tristate register
	bcf     STATUS,RP0      ; Back to page 0
	clrf    CARD_STATE      ; Clear state machine ptr

	BL_OFF			; Turn off Backlighter
	LCDINIT			; Init display

; Print boot up message at start
	LINE1
	PRTSTR	BOOTMSG1
	LINE2
	PRTSTR	BOOTMSG2

	bcf     CINFO,BFR_FULL	; Reset semaphore before checking it out
MAIN:
	btfss   CINFO,BFR_FULL	; Wait for data in buffer
	goto    MAIN
	bcf     CINFO,BFR_FULL	; Reset semaphore

	BL_ON			; When a card is read we want to
	movlw	d'50'		; acknowledge it by blinking with
	call	WAIT		; the backlighter for 100 ms
	BL_OFF

	btfss	CINFO,FAIL	; If fail bit not set
	goto    DISPMSG		; then Read OK, print it out

	LINE1			; Else print the error msg
	PRTSTR	ERRORMSG1	; Small display makes it a twoliner
	LINE2
	PRTSTR	ERRORMSG2
	goto	MAIN		; Wait for next reading

DISPMSG: 
	LINE1			; Start over on the first line

	movlw   BFR_START       ; Set up buffer pointer
	movwf   FSR             ; so we can do indirect reading from buffer

PRINT_CARD:	; This routine doesn't care if we read < 16 chars
	swapf   INDIR,W         ; Read information from buffer (upper nibble)
	andlw   0x0F            ; Mask away upper nibble
	addlw   '0'             ; Add '0' to get ASCII-code
	call    SENDD           ; Print it on display

	movf    INDIR,W         ; Read information form buffer (lower nibble)
	andlw   0x0F            ; Mask away upper nibble
	addlw   '0'             ; Add '0' to get ASCII-code
	call    SENDD           ; Print it on display

	incf    FSR,F           ; Hope we can read FSR
	movf	FSR,W		; Get actual buffer pointer
	xorlw	BFR_START+4	; Written 8 letters??
	btfss	STATUS,Z	; Skip => Equates => Zero => Z=1
	goto	CONT1		; No we haven't
	LINE2
CONT1:
	movf    FSR,W           ; Get actual buffer pointer
	sublw   BFR_END         ; to see if we passed beyond buffer
	btfsc   STATUS,C        ; negative => read beyond => C=0
	goto    PRINT_CARD      ; We haven't read beyond buffer
	goto    MAIN            ; We have read beyond buffer


; INTERRUPT ROUTINE
; Make sure the statemachine (further down) don't reach 
; a jump in the HIGH part
EOCODE:	
	if (EOCODE < 0xFF)
	org	0x100
	endif

INTMAIN:
	btfss   INTCON,RTIF     ; Check if RTCC interrupt
	goto    CARD_INT        ; If it wasn't from RTCC, maybe cardreader
	bcf     INTCON,RTIF     ; Clear RTC interrupt bit
	decf    TIME,F          ; Decrement multipurpose time register
	decf    TIME_OUT,F      ; Decrement TIME_OUT for card-reading
	btfss   STATUS,Z        ; No TIME_OUT wrap around gives 
	return			; Noones bother
	clrf    CARD_STATE      ; Time out; clearing of cardstate
	bcf	INTCON,INTF	; and clearing of cardstate
	return


CARD_INT:               ; BASED ON A HUGE STATEMACHINE
	btfss   INTCON,INTF     ; Was it INT0 interrupting?
	return                  ; If it wasn't, skit i d
	btfsc   PORTB,0         ; Is clock low??
	return                  ; If the clock was high, we don't bother
	bcf     INTCON,INTF     ; Clear interruptflag
	movlw   HIGH CARD_INT   ; Get high part of current address
	movwf   PCLATH          ; Store higher bits for address
	movlw   d'200'
	movwf   TIME_OUT
	movf    CARD_STATE,W    ; Get current state we want to go to
	addwf   PCL,F           ; Go to current state by computed goto
	goto    SYNC            ;0 Wait for sync according to my idea
	goto    READ_BIT        ;1 First we have four databits
	goto    READ_BIT        ;2
	goto    READ_BIT        ;3
	goto    READ_BIT        ;4
	goto    PARITY_1        ;5 Then a parity bit
	goto    READ_BIT        ;6 And another four bits we should chew
	goto    READ_BIT        ;7
	goto    READ_BIT        ;8
	goto    READ_BIT        ;9
	goto    PARITY_2        ;10 Eat parity and increase buffer pointer
	goto    READ_BIT        ;11 Start searching for end sentinel
	goto    READ_BIT        ;12 and chew everything in the way
	goto    READ_BIT        ;13
	goto    READ_BIT        ;14
	goto    END_SENTINEL    ;15 Check if end sentinel
	goto    READ_BIT        ;16 Get next part which is checksum
	goto    READ_BIT        ;17
	goto    READ_BIT        ;18
	goto    READ_BIT        ;19
	goto    CALC_CSUM       ;20 Calculate checksum


SYNC:
	call    READ_BIT        ; Read in bit
	movf    READ,W          ; Get read data so far
	andlw   b'11111000'     ; We're only interested in upper 5 bits
	xorlw   b'01011000'     ; Has we found start sentinel incl. parity?
	btfsc   STATUS,Z        ; Skip if not 0 => Z=0
	goto	SYNC_RESET	; We did, reset most things before reading data
	decf    CARD_STATE,F    ; It wasn't, remove statejump gotten 
	return                  ; in READ_BIT
SYNC_RESET:
	movlw	b'10110000'	; For calculating the checksum we preload
	movwf	CHECKSUM	; with start sentinel
	movlw	BFR_START	; Make buffer pointer point to beginning
	movwf	BFR_PTR		; of buffer
	bcf	CINFO,FAIL	; Clear errormarking bit
	bcf	CINFO,ES_FOUND	; Clear End Sentinel Found-bit
	movlw	0xFF		; Preload buffer with space
	movwf	BFR_START	; If we happens to fail filling complete
	movwf	BFR_START+1	; buffer with data from card.
	movwf	BFR_START+2
	movwf	BFR_START+3
	movwf	BFR_START+4
	movwf	BFR_START+5
	movwf	BFR_START+6
	movwf	BFR_START+7
	return			; Wait for card numbers to come


READ_BIT:
	bcf     STATUS,C        ; Make sure statusbit is clear first
	btfsc   MR_PORT,MR_DATA ; If we have a high databit in
	goto    READ_BIT_LOW    ; we say it's low thus inverting data
	bsf     STATUS,C        ; else set status bit high 
	incf    CHECKSUM,F      ; Inc number of counted 1's for parity check
READ_BIT_LOW:
	rrf     READ,F          ; Card is coded with LSB first !
	incf    CARD_STATE,F    ; Go to next state in statemachine next time
	return


PARITY_1:
	btfss   MR_PORT,MR_DATA ; If paritybit is low (ie high)
	incf    CHECKSUM,F      ; increase number of counted ones
	btfss   CHECKSUM,0      ; If last bit in checksum is 0
	bsf     CINFO,FAIL      ; we have a parityerrror
	movf    CHECKSUM,W      ; Get checksum,
	andlw   0xF0            ; mask away parity calculation information
	movwf   CHECKSUM        ; and put it back where we found it
	movf    READ,W          ; Get just read nibble
	andlw   0xF0            ; Mask away uninteresting information
	xorwf   CHECKSUM,F      ; xor card-data with LRC checksum
	xorlw	0xF0		; Mask with end sentinel (1111) in most 
				; significant nibble to check for it
	btfss	STATUS,Z	; End Sentinel => Zero => Z=1
	goto	NO_ES		; If it wasn't end sentinel, continue
	bsf	CINFO,ES_FOUND	; Tell the world we have found the end sentinel
	movlw	d'16'		; Now that we have found what we wanted
	movwf	CARD_STATE	; goto state looking checksum
	return			; Don't bother about th rest
NO_ES:	
	incf    CARD_STATE,F    ; Goto next state
	return


PARITY_2:
	call    PARITY_1        ; Do the above first(parity and checksumming)
	btfsc	CINFO,ES_FOUND	; If we found End Sentinel don't bother 
	return			; about the rest.
				; That part is dangerous since we don't save
				; the last nibble when we have found the
				; end sentinel. Only valid when odd number of
				; characters.
	movf    BFR_PTR,W       ; Get buffer pointer
	movwf   FSR             ; Put it in FSR, so INDIR points in buffer
	swapf   READ,W          ; First nibble is in highest nibble
	movwf   INDIR           ; Put data in buffer
	incf    BFR_PTR,F       ; Increase bufferpointer
	movf    BFR_PTR,W       ; Put bufferpointer in W for simple calc
	sublw   BFR_END         ; Subtract buffer end to see if end
				; Negative result if we has passed out buffer
	btfss   STATUS,C        ; C=0 result negative
	return			; Negative, we have read all 16 card digits
	movlw   d'1'		; Else goto start state and read next two
	movwf   CARD_STATE      ; digits from card.
	return                  ; End chewing simple parity


END_SENTINEL:
	call    PARITY_1        ; Parity and checksumming
	btfsc	CINFO,ES_FOUND	; If we didn't found end sentinel (1111)
	return			; continue search for it

	movlw   d'11'           ; If it wasn't end sentinel goto research
	movwf   CARD_STATE      ; for end sentinel
	return


CALC_CSUM:
	call    PARITY_1        ; Parity and checksum for CHECKSUM
	movf    CHECKSUM,W      ; Get checksum for checking
	andlw   0xF0            ; Mask so we only have LRC
	btfss   STATUS,Z        ; If we have non-zero it's an error
	bsf     CINFO,FAIL	; Signal block error
	bsf     CINFO,BFR_FULL	; Signal that we have read it all for this
				; time
	clrf    CARD_STATE      ; Go back to syncstate
	return


; TIMING ROUTINES
LCDWAIT:
WAIT:
	movwf   TIME            ; Load time register from W
WAITLOOP:
	movf    TIME,W          ; If TIME=0 then zero flag = 1
	btfsc   STATUS,Z        ; Check if zero?
	return                  ; Yes, return
	goto    WAITLOOP        ; No, wait some more

	end
